package floweditor.component;

import java.awt.Dimension;
import java.awt.Color;
import java.awt.Point;
import java.awt.Graphics;

import java.util.Enumeration;

import com.ibm.graph.Net;
import com.ibm.graph.ext.visual.VisualVertex; 

import floweditor.util.Log;
/** 
    Diese Klasse stellt ein "default" graphisches Editor fr beliebigen
    Komponenten-Instanzen dar. 
 */
public class DefaultGraphEditor 
    extends CompGraphEditor 
{
	
    Component _component;
    Net       _compgraph = new Net ();
    
    private Point  _last_p;
    private Point  _actual_p;
    
    private int _dimCol;
    private int _dimRow;
    
    private int _nCols;
    private int _nRows;
    private int _cellWidth;
    private int _cellHeight;
    
    final private int lineThickness = 2;
    
    
    /** DefaultGraphEditor ist graphische Oberflche auf der die Teilkomponenten 
	einer Komponente (comp) dargestellt werden. Diese Oberflche wird 
	in nCols-Spalten und nRows-Reihen geteilt, innerhalb der einzelnen 
	resultierenden Zellen darf hchstens eine Komponente sich befinden. 
    */
    public DefaultGraphEditor (Component comp, int nCols, int nRows, 
			    int cellWidth, int cellHeight) {
    
	super (comp);	
	_component = comp;
	
	_dimCol = nCols*cellWidth + (nCols + 1)*lineThickness;
	_dimRow = nRows*cellHeight + (nRows +1)*lineThickness; 	
	
	_cellWidth  = cellWidth;
	_cellHeight = cellHeight;
	_nCols = nCols;
	_nRows = nRows;
	
	setPreferredSize (new Dimension (_dimCol, _dimRow));
	setBackground (Color.white);
	System.err.println ("grapheditor is showing?: " + this.isShowing ());
	
    }

    /**
     *  ber diese Methode wird nach jeder Interaktion, wie z.B. das Entfernen
     *  und/oder das An-/Einfgen eine Teilkomponente die Darstellung angepasst.
     *  "paint" verwendet paintGrid und paintComponents.
     */
    public void paint (Graphics g) 
    {
	super.paint (g);

	paintGrid (g);
	paintComponents (g);
    }
    
     /**
      *  Zeichnet diese Malflche als vergitterte Flche.
      */
    public void paintGrid (Graphics g) 
    {
 	for (int col = 0; col <= (_nCols + 1); col++) {
 	    int posCol = col*(_cellWidth + lineThickness);
 	    g.drawLine (posCol, 0, posCol, _dimRow);
 	}
	
 	for (int row = 0; row <= (_nRows + 1); row++) {
 	    int posRow = row*(_cellHeight + lineThickness);
 	    g.drawLine (0, posRow, _dimCol, posRow);
 	}
    }
    
    /**
     *  Zeichnet die Teilkomponente eines Elementes (bzw. einer Komponente)
     *  auf der Malflache. Jede Teilkomponente wird durch ihre graphische 
     *  Instanz, eine graphische Instanz wird jeweils innerhalb einer Zelle 
     *  dargestellt.
     */
    public void paintComponents (Graphics g) 
    {
	_compgraph.clear ();
	Enumeration components = _component.elements ();
	
	for (int i = 0; (components.hasMoreElements () && i < _nRows); i++) {
	    for (int j = 0; (components.hasMoreElements () && j < _nCols); j++) {
		Point cellCentre = getCellCentre (j, i);
		
		Component    childComponent = 
		    (Component) components.nextElement ();
		DefaultGraph childGraph     = 
		    (DefaultGraph) childComponent.getCompGraph ();
		if (childGraph.isVertex ()) {

		    childGraph.setLocation (cellCentre);
		    _compgraph.add (childGraph);
		} else {  // Komponente ist Transition
		    VisualVertex transGraph = 
			new VisualVertex (VisualVertex.ELLIPSE,  
					  Color.red, Color.black,
					  false, "TRANSITION",
					  Color.black);    
		    _compgraph.add (transGraph);
		}
	    }
	}
	_compgraph.draw (g);
    }

    /** Methode aktualisiert Darstellung, falls neue Kinder-Komponenten
	hinzugekommen/entfernt sind.
     */
    public void update () 
    {
	_compgraph.clear ();
	if (this.isShowing ()) {
	    repaint ();
	}
    }


     /** Gibt den Mittelpunkt (die Koordinaten) der Zelle auf der 
      *  Position Reihe col, Zeile row.
      */
     public Point getCellCentre (int col, int row) 
     {
 	int colCenter 
 	    = col*(_cellWidth + lineThickness)
 	    + (_cellWidth / 2)
 	    + (lineThickness / 2);
 	int rowCenter 
 	    = row*(_cellHeight + lineThickness) 
 	    + (_cellHeight / 2)
 	    + (lineThickness / 2);
	
 	return new Point (colCenter, rowCenter);
     }
}











